/**
 * Icon Generation Script
 * Generates PNG icons from SVG for Chrome Extension
 * 
 * Run: node extension/icons/generate-icons.js
 * Requires: npm install sharp
 */

const fs = require('fs');
const path = require('path');

// Read the actual brand logo SVG
const logoSvgPath = path.join(__dirname, '..', '..', 'public', 'icon.svg');
let svgContent;

try {
  svgContent = fs.readFileSync(logoSvgPath, 'utf8');
  console.log('✅ Using brand logo from public/icon.svg');
} catch (e) {
  console.log('⚠️  Brand logo not found, using fallback SVG');
  // Fallback to simple Z logo
  svgContent = `<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 128 128">
    <defs>
      <linearGradient id="grad" x1="0%" y1="0%" x2="100%" y2="100%">
        <stop offset="0%" style="stop-color:#7c3aed;stop-opacity:1" />
        <stop offset="100%" style="stop-color:#4f46e5;stop-opacity:1" />
      </linearGradient>
    </defs>
    <rect width="128" height="128" rx="24" fill="url(#grad)"/>
    <text x="64" y="88" font-family="Arial, sans-serif" font-size="72" font-weight="bold" fill="white" text-anchor="middle">Z</text>
  </svg>`;
}

const sizes = [16, 32, 48, 128];

async function generateIcons() {
  try {
    let sharp;
    try {
      sharp = require('sharp');
    } catch (e) {
      console.log(' sharp not installed. Installing...');
      console.log(' Please run: npm install sharp');
      console.log('');
      console.log(' Alternative: Use an online SVG to PNG converter');
      process.exit(1);
    }

    const svgBuffer = Buffer.from(svgContent);
    
    for (const size of sizes) {
      const outputPath = path.join(__dirname, `icon${size}.png`);
      await sharp(svgBuffer)
        .resize(size, size)
        .png()
        .toFile(outputPath);
      console.log(`Generated: icon${size}.png (${size}x${size})`);
    }
    
    console.log('\n✅ All icons generated successfully!');
    console.log('📁 Location: extension/icons/');
    console.log('🔄 Commit and push these changes to GitHub');
  } catch (error) {
    console.error('Error generating icons:', error);
    process.exit(1);
  }
}

generateIcons();
