/**
 * ZDRAVO AI - Offline Extension Installer
 * Provides full functionality without requiring login
 * Syncs when user authenticates
 */

(function() {
  const CONFIG = {
    API_URL: window.location.origin.includes('localhost') 
      ? 'http://localhost:3000' 
      : (process?.env?.NEXT_PUBLIC_APP_URL || 'https://www.zdravoai.com'),
    OFFLINE_STORAGE_KEY: 'zdravo_offline_memories',
    SYNC_QUEUE_KEY: 'zdravo_sync_queue',
    USER_SETTINGS_KEY: 'zdravo_user_settings',
    VERSION: '2.0.0'
  };

  // Prevent multiple injections
  if (document.getElementById('zdravo-extension-root')) {
    console.log('ZDRAVO Extension already loaded');
    return;
  }

  // Initialize extension
  const ZdravoExtension = {
    isOnline: navigator.onLine,
    isAuthenticated: false,
    offlineMemories: [],
    syncQueue: [],

    // Initialize extension
    init() {
      console.log('🚀 Initializing ZDRAVO Extension v' + CONFIG.VERSION);
      
      this.loadOfflineData();
      this.setupBookmarklet();
      this.setupContextMenu();
      this.setupSyncIndicator();
      this.setupAutoSync();
      
      // Check authentication status
      this.checkAuthStatus();
      
      // Setup online/offline listeners
      window.addEventListener('online', () => this.handleOnline());
      window.addEventListener('offline', () => this.handleOffline());
      
      console.log('✅ ZDRAVO Extension ready');
    },

    // Load offline data
    loadOfflineData() {
      try {
        this.offlineMemories = JSON.parse(localStorage.getItem(CONFIG.OFFLINE_STORAGE_KEY) || '[]');
        this.syncQueue = JSON.parse(localStorage.getItem(CONFIG.SYNC_QUEUE_KEY) || '[]');
        console.log(`📦 Loaded ${this.offlineMemories.length} offline memories, ${this.syncQueue.length} items in sync queue`);
      } catch (error) {
        console.error('Error loading offline data:', error);
        this.offlineMemories = [];
        this.syncQueue = [];
      }
    },

    // Save offline data
    saveOfflineData() {
      try {
        localStorage.setItem(CONFIG.OFFLINE_STORAGE_KEY, JSON.stringify(this.offlineMemories));
        localStorage.setItem(CONFIG.SYNC_QUEUE_KEY, JSON.stringify(this.syncQueue));
      } catch (error) {
        console.error('Error saving offline data:', error);
      }
    },

    // Check authentication status
    async checkAuthStatus() {
      try {
        const response = await fetch(`${CONFIG.API_URL}/api/auth/user`, {
          method: 'GET',
          credentials: 'include'
        });
        
        this.isAuthenticated = response.ok;
        this.updateSyncIndicator();
        
        if (this.isAuthenticated && this.syncQueue.length > 0) {
          console.log(`🔄 Syncing ${this.syncQueue.length} queued items...`);
          await this.processSyncQueue();
        }
      } catch (error) {
        this.isAuthenticated = false;
        this.updateSyncIndicator();
      }
    },

    // Setup enhanced bookmarklet
    setupBookmarklet() {
      // Enhanced bookmarklet button
      const bookmarkletBtn = document.createElement('div');
      bookmarkletBtn.id = 'zdravo-bookmarklet-btn';
      bookmarkletBtn.innerHTML = `
        <div class="zdravo-btn-content">
          <img src="${CONFIG.API_URL}/favicon.ico" class="zdravo-btn-logo" alt="Z">
          <span class="zdravo-btn-text">ZDRAVO</span>
          <span class="zdravo-sync-indicator" id="zdravo-sync-indicator"></span>
        </div>
      `;
      
      // Add styles
      const style = document.createElement('style');
      style.textContent = `
        #zdravo-bookmarklet-btn {
          position: fixed;
          top: 20px;
          right: 20px;
          z-index: 2147483647;
          cursor: pointer;
          transition: all 0.3s ease;
        }
        .zdravo-btn-content {
          display: flex;
          align-items: center;
          gap: 8px;
          padding: 10px 16px;
          background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
          color: white;
          border-radius: 25px;
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
          font-weight: 600;
          font-size: 14px;
          box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
          border: 2px solid rgba(255, 255, 255, 0.2);
        }
        #zdravo-bookmarklet-btn:hover {
          transform: translateY(-2px);
          box-shadow: 0 6px 20px rgba(102, 126, 234, 0.5);
        }
        .zdravo-btn-logo {
          width: 20px;
          height: 20px;
          border-radius: 50%;
        }
        .zdravo-btn-text {
          font-size: 13px;
          font-weight: 700;
          letter-spacing: 0.5px;
        }
        .zdravo-sync-indicator {
          width: 8px;
          height: 8px;
          border-radius: 50%;
          background: #10b981;
          animation: pulse 2s infinite;
        }
        .zdravo-sync-indicator.offline {
          background: #f59e0b;
          animation: none;
        }
        .zdravo-sync-indicator.error {
          background: #ef4444;
          animation: none;
        }
        @keyframes pulse {
          0%, 100% { opacity: 1; }
          50% { opacity: 0.5; }
        }
        .zdravo-offline-badge {
          position: absolute;
          top: -8px;
          right: -8px;
          background: #ef4444;
          color: white;
          border-radius: 50%;
          width: 20px;
          height: 20px;
          display: flex;
          align-items: center;
          justify-content: center;
          font-size: 11px;
          font-weight: bold;
        }
      `;
      document.head.appendChild(style);
      
      document.body.appendChild(bookmarkletBtn);
      
      // Click handler
      bookmarkletBtn.addEventListener('click', () => this.captureCurrentPage());
    },

    // Setup enhanced context menu
    setupContextMenu() {
      document.addEventListener('contextmenu', (e) => {
        const selectedText = window.getSelection().toString().trim();
        if (selectedText) {
          e.preventDefault();
          this.showContextMenu(e.clientX, e.clientY, selectedText);
        }
      });
    },

    // Show context menu
    showContextMenu(x, y, selectedText) {
      // Remove existing menu
      const existing = document.getElementById('zdravo-context-menu');
      if (existing) existing.remove();

      const menu = document.createElement('div');
      menu.id = 'zdravo-context-menu';
      menu.innerHTML = `
        <div class="zdravo-context-item" data-action="save-selection">
          <img src="${CONFIG.API_URL}/favicon.ico" class="zdravo-context-logo" alt="Z">
          <span>Save to ZDRAVO</span>
          <span class="zdravo-context-status">${this.isAuthenticated ? 'Online' : 'Offline'}</span>
        </div>
        <div class="zdravo-context-divider"></div>
        <div class="zdravo-context-item" data-action="view-memories">
          <span>📚</span>
          <span>View Saved (${this.offlineMemories.length})</span>
        </div>
        <div class="zdravo-context-item" data-action="sync-now" ${!this.isAuthenticated ? 'style="display:none"' : ''}>
          <span>🔄</span>
          <span>Sync Now</span>
        </div>
      `;

      // Add styles
      const style = document.createElement('style');
      style.textContent = `
        #zdravo-context-menu {
          position: fixed;
          background: white;
          border: 1px solid #e2e8f0;
          border-radius: 12px;
          box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
          z-index: 2147483647;
          min-width: 220px;
          animation: zdravoFadeIn 0.2s ease-out;
          backdrop-filter: blur(10px);
        }
        .zdravo-context-item {
          display: flex;
          align-items: center;
          gap: 10px;
          padding: 12px 16px;
          cursor: pointer;
          font-size: 14px;
          color: #374151;
          transition: all 0.15s;
          border-radius: 8px;
          margin: 4px;
        }
        .zdravo-context-item:hover {
          background: #f8fafc;
          transform: translateX(2px);
        }
        .zdravo-context-logo {
          width: 18px;
          height: 18px;
          border-radius: 4px;
        }
        .zdravo-context-status {
          margin-left: auto;
          font-size: 11px;
          padding: 2px 6px;
          border-radius: 10px;
          background: #10b981;
          color: white;
          font-weight: 500;
        }
        .zdravo-context-divider {
          height: 1px;
          background: #e5e7eb;
          margin: 4px 12px;
        }
      `;
      document.head.appendChild(style);

      menu.style.left = x + 'px';
      menu.style.top = y + 'px';
      document.body.appendChild(menu);

      // Handle clicks
      menu.addEventListener('click', (e) => {
        const action = e.target.closest('.zdravo-context-item')?.dataset.action;
        if (action) {
          menu.remove();
          this.handleContextAction(action, selectedText);
        }
      });

      // Close on outside click
      setTimeout(() => {
        document.addEventListener('click', function closeMenu(e) {
          if (!menu.contains(e.target)) {
            menu.remove();
            document.removeEventListener('click', closeMenu);
          }
        });
      }, 100);
    },

    // Handle context menu actions
    handleContextAction(action, selectedText) {
      switch (action) {
        case 'save-selection':
          this.saveSelection(selectedText);
          break;
        case 'view-memories':
          this.showMemoriesViewer();
          break;
        case 'sync-now':
          this.forceSync();
          break;
      }
    },

    // Save selection (offline or online)
    async saveSelection(content) {
      const memory = {
        id: Date.now().toString(),
        title: this.generateTitle(content),
        content: content,
        url: window.location.href,
        platform: this.detectPlatform(),
        timestamp: new Date().toISOString(),
        tags: [this.detectPlatform().toLowerCase(), 'context-menu'],
        synced: false
      };

      // Save offline immediately
      this.offlineMemories.unshift(memory);
      this.saveOfflineData();

      if (this.isAuthenticated && this.isOnline) {
        // Try to sync immediately
        await this.syncMemory(memory);
      } else {
        // Add to sync queue
        this.syncQueue.push(memory.id);
        this.saveOfflineData();
        this.showNotification('Saved offline! Will sync when you log in.', 'success');
      }
    },

    // Capture current page
    async captureCurrentPage() {
      const content = await this.extractPageContent();
      const memory = {
        id: Date.now().toString(),
        title: document.title,
        content: content,
        url: window.location.href,
        platform: this.detectPlatform(),
        timestamp: new Date().toISOString(),
        tags: [this.detectPlatform().toLowerCase(), 'full-page'],
        synced: false
      };

      this.offlineMemories.unshift(memory);
      this.saveOfflineData();

      if (this.isAuthenticated && this.isOnline) {
        await this.syncMemory(memory);
      } else {
        this.syncQueue.push(memory.id);
        this.saveOfflineData();
        this.showNotification('Page saved offline! Will sync when you log in.', 'success');
      }
    },

    // Extract page content
    async extractPageContent() {
      // Platform-specific extraction
      const platform = this.detectPlatform();
      
      switch (platform) {
        case 'ChatGPT':
          return this.extractChatGPTContent();
        case 'Claude':
          return this.extractClaudeContent();
        case 'Gemini':
          return this.extractGeminiContent();
        default:
          return this.extractGenericContent();
      }
    },

    // Platform-specific extractors
    extractChatGPTContent() {
      const messages = document.querySelectorAll('[data-message-author-role]');
      return Array.from(messages).map(msg => {
        const role = msg.getAttribute('data-message-author-role') === 'user' ? 'You' : 'ChatGPT';
        return `${role}: ${msg.textContent?.trim() || ''}`;
      }).join('\n\n');
    },

    extractClaudeContent() {
      const messages = document.querySelectorAll('.font-claude-message, .font-user-message');
      return Array.from(messages).map(msg => {
        const role = msg.classList.contains('font-user-message') ? 'You' : 'Claude';
        return `${role}: ${msg.textContent?.trim() || ''}`;
      }).join('\n\n');
    },

    extractGeminiContent() {
      const messages = document.querySelectorAll('.message-content');
      return Array.from(messages).map((msg, i) => {
        const role = i % 2 === 0 ? 'You' : 'Gemini';
        return `${role}: ${msg.textContent?.trim() || ''}`;
      }).join('\n\n');
    },

    extractGenericContent() {
      // Try to get main content area
      const selectors = [
        'main', 'article', '[role="main"]', '.content', '#content',
        'body'
      ];
      
      for (const selector of selectors) {
        const element = document.querySelector(selector);
        if (element) {
          return element.textContent?.trim().substring(0, 10000) || '';
        }
      }
      
      return document.body.textContent?.trim().substring(0, 10000) || '';
    },

    // Sync memory to server
    async syncMemory(memory) {
      if (!this.isAuthenticated || !this.isOnline) return false;

      try {
        const response = await fetch(`${CONFIG.API_URL}/api/memories`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          credentials: 'include',
          body: JSON.stringify({
            title: memory.title,
            content: memory.content,
            url: memory.url,
            sourcePlatform: memory.platform.toLowerCase(),
            tags: memory.tags
          })
        });

        if (response.ok) {
          // Mark as synced
          memory.synced = true;
          memory.serverId = (await response.json()).id;
          
          // Update offline storage
          const index = this.offlineMemories.findIndex(m => m.id === memory.id);
          if (index !== -1) {
            this.offlineMemories[index] = memory;
          }
          
          // Remove from sync queue
          this.syncQueue = this.syncQueue.filter(id => id !== memory.id);
          this.saveOfflineData();
          
          return true;
        }
      } catch (error) {
        console.error('Sync error:', error);
        return false;
      }
      
      return false;
    },

    // Process sync queue
    async processSyncQueue() {
      if (!this.isAuthenticated || !this.isOnline) return;

      const queueToProcess = [...this.syncQueue];
      
      for (const memoryId of queueToProcess) {
        const memory = this.offlineMemories.find(m => m.id === memoryId);
        if (memory && !memory.synced) {
          const synced = await this.syncMemory(memory);
          if (synced) {
            this.showNotification('Synced: ' + memory.title, 'success');
          }
        }
      }
    },

    // Show memories viewer
    showMemoriesViewer() {
      // Create modal to view offline memories
      const modal = document.createElement('div');
      modal.innerHTML = `
        <div class="zdravo-viewer-overlay">
          <div class="zdravo-viewer">
            <div class="zdravo-viewer-header">
              <h3>📚 Your ZDRAVO Memories</h3>
              <button class="zdravo-viewer-close">✕</button>
            </div>
            <div class="zdravo-viewer-content">
              <div class="zdravo-viewer-stats">
                <span>Total: ${this.offlineMemories.length}</span>
                <span>Synced: ${this.offlineMemories.filter(m => m.synced).length}</span>
                <span>Queued: ${this.syncQueue.length}</span>
              </div>
              <div class="zdravo-memories-list">
                ${this.offlineMemories.slice(0, 10).map(memory => `
                  <div class="zdravo-memory-item ${memory.synced ? 'synced' : 'pending'}">
                    <div class="zdravo-memory-header">
                      <h4>${memory.title}</h4>
                      <span class="zdravo-memory-status">${memory.synced ? '✓' : '⏳'}</span>
                    </div>
                    <p class="zdravo-memory-preview">${memory.content.substring(0, 150)}...</p>
                    <div class="zdravo-memory-meta">
                      <span>${memory.platform}</span>
                      <span>${new Date(memory.timestamp).toLocaleDateString()}</span>
                    </div>
                  </div>
                `).join('')}
              </div>
            </div>
          </div>
        </div>
      `;

      // Add styles
      const style = document.createElement('style');
      style.textContent = `
        .zdravo-viewer-overlay {
          position: fixed;
          top: 0;
          left: 0;
          right: 0;
          bottom: 0;
          background: rgba(0, 0, 0, 0.7);
          z-index: 2147483647;
          display: flex;
          align-items: center;
          justify-content: center;
        }
        .zdravo-viewer {
          background: white;
          border-radius: 16px;
          width: 600px;
          max-width: 90vw;
          max-height: 80vh;
          overflow: hidden;
        }
        .zdravo-viewer-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 20px;
          border-bottom: 1px solid #e5e7eb;
        }
        .zdravo-viewer-header h3 {
          margin: 0;
          font-size: 18px;
          font-weight: 600;
        }
        .zdravo-viewer-close {
          background: none;
          border: none;
          font-size: 20px;
          cursor: pointer;
          padding: 4px;
          border-radius: 4px;
        }
        .zdravo-viewer-content {
          padding: 20px;
          max-height: 60vh;
          overflow-y: auto;
        }
        .zdravo-viewer-stats {
          display: flex;
          gap: 20px;
          margin-bottom: 20px;
          font-weight: 500;
        }
        .zdravo-memories-list {
          display: flex;
          flex-direction: column;
          gap: 12px;
        }
        .zdravo-memory-item {
          border: 1px solid #e5e7eb;
          border-radius: 8px;
          padding: 12px;
        }
        .zdravo-memory-item.synced {
          border-color: #10b981;
          background: #f0fdf4;
        }
        .zdravo-memory-item.pending {
          border-color: #f59e0b;
          background: #fffbeb;
        }
        .zdravo-memory-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          margin-bottom: 8px;
        }
        .zdravo-memory-header h4 {
          margin: 0;
          font-size: 14px;
          font-weight: 600;
        }
        .zdravo-memory-status {
          font-size: 12px;
          font-weight: bold;
        }
        .zdravo-memory-preview {
          margin: 8px 0;
          font-size: 13px;
          color: #6b7280;
          line-height: 1.4;
        }
        .zdravo-memory-meta {
          display: flex;
          gap: 12px;
          font-size: 11px;
          color: #9ca3af;
        }
      `;
      document.head.appendChild(style);

      document.body.appendChild(modal);

      // Event listeners
      modal.querySelector('.zdravo-viewer-close').onclick = () => modal.remove();
      modal.querySelector('.zdravo-viewer-overlay').onclick = (e) => {
        if (e.target === modal.querySelector('.zdravo-viewer-overlay')) {
          modal.remove();
        }
      };
    },

    // Update sync indicator
    updateSyncIndicator() {
      const indicator = document.getElementById('zdravo-sync-indicator');
      if (!indicator) return;

      if (!this.isOnline) {
        indicator.className = 'zdravo-sync-indicator offline';
      } else if (!this.isAuthenticated) {
        indicator.className = 'zdravo-sync-indicator error';
      } else if (this.syncQueue.length > 0) {
        indicator.className = 'zdravo-sync-indicator';
      } else {
        indicator.className = 'zdravo-sync-indicator';
      }

      // Add badge for queued items
      const badge = indicator.parentElement.querySelector('.zdravo-offline-badge');
      if (this.syncQueue.length > 0 && !badge) {
        const badgeEl = document.createElement('div');
        badgeEl.className = 'zdravo-offline-badge';
        badgeEl.textContent = this.syncQueue.length;
        indicator.parentElement.appendChild(badgeEl);
      } else if (this.syncQueue.length === 0 && badge) {
        badge.remove();
      }
    },

    // Setup auto-sync
    setupAutoSync() {
      // Try to sync every 30 seconds when authenticated
      setInterval(async () => {
        if (this.isAuthenticated && this.isOnline && this.syncQueue.length > 0) {
          await this.processSyncQueue();
        }
      }, 30000);
    },

    // Handle online/offline
    handleOnline() {
      this.isOnline = true;
      this.updateSyncIndicator();
      if (this.isAuthenticated && this.syncQueue.length > 0) {
        this.processSyncQueue();
      }
      this.showNotification('Back online! Syncing queued items...', 'success');
    },

    handleOffline() {
      this.isOnline = false;
      this.updateSyncIndicator();
      this.showNotification('Offline mode - memories will sync when online', 'warning');
    },

    // Force sync
    async forceSync() {
      if (!this.isAuthenticated) {
        this.showNotification('Please log in to sync', 'error');
        return;
      }

      if (!this.isOnline) {
        this.showNotification('No internet connection', 'error');
        return;
      }

      this.showNotification('Syncing...', 'info');
      await this.processSyncQueue();
    },

    // Show notification
    showNotification(message, type = 'info') {
      const notification = document.createElement('div');
      notification.className = `zdravo-notification zdravo-${type}`;
      notification.textContent = message;

      const style = document.createElement('style');
      style.textContent = `
        .zdravo-notification {
          position: fixed;
          bottom: 20px;
          right: 20px;
          padding: 12px 20px;
          border-radius: 8px;
          font-size: 14px;
          font-weight: 500;
          z-index: 2147483647;
          animation: slideIn 0.3s ease-out;
        }
        .zdravo-success { background: #10b981; color: white; }
        .zdravo-error { background: #ef4444; color: white; }
        .zdravo-warning { background: #f59e0b; color: white; }
        .zdravo-info { background: #3b82f6; color: white; }
        @keyframes slideIn {
          from { transform: translateX(100%); opacity: 0; }
          to { transform: translateX(0); opacity: 1; }
        }
      `;
      document.head.appendChild(style);

      document.body.appendChild(notification);

      setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease-in forwards';
        setTimeout(() => notification.remove(), 300);
      }, 3000);
    },

    // Utility functions
    generateTitle(content) {
      const platform = this.detectPlatform();
      const preview = content.substring(0, 50).replace(/\n/g, ' ');
      return `${platform} Selection - ${preview}${content.length > 50 ? '...' : ''}`;
    },

    detectPlatform() {
      const url = window.location.href;
      if (url.includes('chatgpt.com') || url.includes('chat.openai.com')) return 'ChatGPT';
      if (url.includes('claude.ai')) return 'Claude';
      if (url.includes('gemini.google.com')) return 'Gemini';
      if (url.includes('perplexity.ai')) return 'Perplexity';
      return 'Web Page';
    }
  };

  // Initialize extension
  ZdravoExtension.init();
  
  // Make available globally for debugging
  window.ZdravoExtension = ZdravoExtension;
  
  // Mark as loaded
  const marker = document.createElement('div');
  marker.id = 'zdravo-extension-root';
  marker.style.display = 'none';
  document.body.appendChild(marker);
})();
